<?php
/**
* @package icarus
* @subpackage libraries
*/

load_libraries('Object', 'form_markup_generator/Field_markup_generator');
load_helpers('array');

/**
* Get the path to a field markup generator.
* Will return the correct path to any application-specific extensions.
* @param string Field markup generator type.
*/
function field_markup_generator_path($type, $extension_directory=NULL){
	if(!validates_as('nonempty_string', $type)) return should_be('nonempty_string', $type);
	if(!is_null($extension_directory) && !validates_as('directory_path', $extension_directory)) return should_be('directory path', $directory_path);
	
	if(defined('CI_VERSION')){ //if we're in CI, use the loader to make sure we get any extensions
		$CI = &get_instance();
		$CI->load->library('form_markup_generator');
		return $CI->form_markup_generator->path_for_field_type($type);
	}
	
	$form_markup_generator = new Form_markup_generator();
	if(!is_null($extension_directory))
		$form_markup_generator->extension_directory = $extension_directory;
	return $form_markup_generator->path_for_field_type($type);
}

/**
* @package icarus
* @subpackage libraries
*/
class Form_markup_generator extends Object{	
	/**
	* @var string Model class name (or default alias -- whatever you need to be able to access it via the controller)
	*/
	protected $_model_class; //if this is set, each field for this form will be considered to be from this model
	protected $_fields = array(); //$name => Field_markup_generator	
	
	//The Icarus form markup generator info is located in ICARUS_DIRECTORY.libraries/form_markup_generator.
	//If you have any application-specific extensions, put your application's equivalent here.  (default for Codeigniter: APPATH.libraries/form_markup_generator)
	protected $_extension_directory; //add a setter to make sure that this is a real directory
	
	
	protected $_property_validation_rules= array(	'extension_directory' => 'directory_path');			
	
	function __construct($fields = array()){
		parent::__construct();
		if(!is_null($fields)){
			$this->fields = $fields;
		}
	}
	
	function validates(){
		if($this->property_is_empty('fields')) return $this->error->warning("I can't validate a form that has no fields.");
		$form_validates = true;
		foreach($this->fields as $field){
			$form_validates = ($field->validates() && $form_validates);
		}
		return $form_validates;
	}
	
	function validation_messages(){
		$messages = array();
		foreach($this->fields as $field){
			$message = $field->validation_messages();
			if(!empty($message)) $messages[] = $message;
		}
		return $messages;
	}
	
///////////////////////////
// GETTERS FOR CLASS VARS
////////////////////////////
	
	function extension_directory(){
		$return_value = null;
		if(isset($this->_extension_directory))
			$return_value = $this->_extension_directory;
		elseif(defined('CI_VERSION') && defined('APPPATH') && file_exists(APPPATH.'libraries/form_markup_generator')) 
			$return_value = APPPATH.'libraries/form_markup_generator/';
		
		if(!is_null($return_value) && !string_ends_with('/', $return_value))
			$return_value .= '/';
		
		return $return_value;
	}

		
///////////////////////////
// SETTERS FOR CLASS VARS
///////////////////////////
	
	/**
	* Setter for {@link _model_class}.
	* @param string $model_name
	*/
	function set_model_class($model_name){
		if(!$this->validator->is_a_model_class($model_name)) return $this->error->property_value_should_be_a_model_class_name('model_class', $this, $model_name);
		$this->_model_class = $model_name;
		if(!$this->property_is_empty('fields')){
			foreach($this->field_names as $name){
				$this->set_field_property($name, 'model_class', $this->model_class);
			}
		}
	}
	

	/**
	* Setter for {@link _fields}.
	* The parameter should be an array in which the keys are the field names and the values are either the field type or
	* an array containing the field type and additional field configuration options.
	* @param array $fields
	*/
	function set_fields($fields){
		if(!$this->validator->is_an_array($fields)) return $this->error->property_value_should_be_an_array('fields', $this, $fields);
		foreach($fields as $name => $type_or_values){
			$this->set_field($name, $type_or_values);
		}		
	}
	
/////////////////////////////////////
/// FIELD FUNCTIONS
/////////////////////////////////////	
	
	/**
	* Create a field markup generator and store it in {@_fields}.
	* @param name Name of the field (ie, the "name" HTML attribute of the form field tag.)
	* @param string|array Either the type of field markup generator, or an array that contains the type & other values for the field markup generator.
	*/
	function set_field($name, $type_or_values){		
		if(!$this->validator->is_a_nonempty_string($name)) return $this->error->should_be_a_nonempty_string($name);
		if(!$this->is_a_valid_field_type($type_or_values) && !$this->validator->is_an_array($type_or_values))
			return should_be('a valid form field type or an array for the '.$name.' field', $type_or_values);

		$type = $type_or_values;
		$values = array();
		if(is_array($type_or_values)){
			$type = element('type', $type_or_values);
			$values = $type_or_values;
			unset($values['type']);
		}
		
		if(!$this->is_a_valid_field_type($type)) return $this->error->should_be_a_valid_form_field_type($type);
		$field_markup_file = $this->path_for_field_type($type);
		$extension_directory = $this->extension_directory;

 		$properties_to_pass_on = array('model_class', 'extension_directory');
		foreach($properties_to_pass_on as $property){
				if(!$this->property_is_empty($property) && !array_key_exists($property, $values)){
					if(method_exists(get_class($this), $property))
						$values[$property] = $this->$property();
					else
						$values[$property] = $this->$property;
				}
				if(array_key_exists($property, $values) && $values[$property] == null) unset($values[$property]);
		}

		require_once($field_markup_file);
		$class = ucfirst($type).'_markup_generator';
		$field_markup_generator = new $class($name, $values);
		$this->_fields[$name] = &$field_markup_generator; 
	}
	
	/**
	* Set the value for multiple fields at once.
	* Ideal for loading values into the form from $_POST or a db array.
	*
	* @uses set_field_property
	*
	* @param array The keys should be the field names, the values should be the field values.
	*/
	function set_values($values){
		if(!$this->validator->is_an_array($values)) return $this->error->should_be_an_array($values);
		foreach($values as $field_name => $value){
			$this->set_field_property($field_name, 'value', $value);
		}
	}
	
	function set_values_from_post(){	
		
		//for most values, we just need to figure out which fields from this markup generator are currently in POS   n  set them
		$values_from_post = array_intersect_key($_POST, $this->fields);
		$this->set_values($values_from_post);
		
		//field names with '[' in them won't show up correctly from the easy code above, so we'll handle them seperately
		foreach(array_diff($this->field_names(), array_keys($_POST)) as $field_name){
			if(!string_contains('[', $field_name)) continue;  //if a field doesn't have a ']', it wasn't in POST because it wasn't submitted
			
			//our field name is an actual string "field[subfield][any-number-of-subfields]" - take it apart so that we can figure out all the array keys that we need to grab		
			$pieces = explode('[', $field_name);
			$pieces = array_map('strip_from_end', array_fill(0, count($pieces), ']'), $pieces);
			
			//drill down to the level indicated by the field name - this is turning "field[subfield][any-number-of-subfields]" into $_POST['field']['subfield']['any-number-of-subfields']
			$value = $_POST;
			foreach($pieces as $piece)
				$value = element($piece, $value);
			
			$this->field($field_name)->value = $value;
		}		
	}
	
	function set_required_fields($field_names){
		if(!$this->validator->is_an_array_of_nonempty_strings($field_names)) return $this->error->should_be_an_array_of_nonempty_strings($field_names);
		foreach($field_names as $field_name){
			$this->set_field_property($field_name, 'required', true);
		}
	}
	
	/**
	* Sets a property on the markup generator for a given field of this form.
	* @param string The name of the form field (as listed in {@link _fields}
	* @param string The name of the property to set.
	* @param mixed The value of the property.
	*/
	function set_field_property($field_name, $property, $value){
		if(!$this->field_exists($field_name)) return $this->error->should_be_the_name_of_an_existing_field($field_name);
		$this->field($field_name)->$property = $value;
	}
	
	function markup(){
		return implode("\n", $this->markup_for_fields());
	}
	
	function markup_for_fields(){
		$markup = array();
		foreach($this->fields as $name => $field)
			$markup[$name] = $field->markup();
		return $markup;
	}
	
	function markup_for_field($name){
		if(!$this->field_exists($name)) return $this->error->should_be_the_name_of_an_existing_field($name);
		return $this->field($name)->markup();
	}
	
	function &field($name){
		if(!$this->validator->is_a_nonempty_string($name) || !$this->validator->is_a_string_with_no_whitespace($name))
			return $this->error->should_be_a_nonempty_string_with_no_spaces($name);
		if(!array_key_exists($name, $this->fields))	
			return $this->error->warning('There is no field named '.$name);
		
		$field = &element($name, $this->fields);
		return $field;
	}
	
	function field_exists($name){
		$fields = $this->fields;
		return array_key_exists($name, $this->fields) && is_a($fields[$name], 'Field_markup_generator');
	}
	
	function value($field_name){
		if(!$this->field_exists($field_name)) return $this->error->should_be_an_existing_form_field_name($field_name);
		return $this->field($field_name)->value;
	}
	
	function values($just_these_names = null){
		if(is_null($just_these_names)) $just_these_names = $this->field_names();
		if(!$this->validator->is_an_array_of_nonempty_string_with_no_whitespaces($just_these_names)) 
			return $this->error->should_be_null_or_an_array_of_nonempty_strings_with_no_whitespace($just_these_names);
		$values = array();
		foreach($just_these_names as $field_name){
			$values[$field_name] = $this->value($field_name);
		}
		return $values;
	}
		
	//field types used in this particular forms
	function field_types(){
		$types = array();
		foreach($this->fields as $field){
			if(!in_array($field->type, $types))
				$types[] = $field->type;
		}
		return $types;
	}
	
		/**
	* Names of all the fields currently loaded into this field.
	* @return array
	*/
	function field_names(){
		return array_keys($this->fields);
	}
	
	function field_names_for_markup(){
		$names_for_markup = array();
		foreach($this->fields as $name => $field){
			$names_for_markup[$name] = $field->name_for_markup();
		}
		return $names_for_markup;
	}
	
	function fields_for_model($model_class=null){
		if(is_null($model_class) && !$this->property_is_empty('model_class')) $model_class = $this->model_class;
		if(!$this->validator->is_a_model_class($model_class)) return $this->error->should_be_a_model_class($model_class);
		$fields_for_model = array();
		foreach($this->fields as $field_name => $field){
			if(!$field->property_is_empty('model_class') && $field->model_class == $model_class)
				$fields_for_model[$field_name] = $field;
		}
		return $fields_for_model;
	}
	
	function field_names_for_model($model_class=null){
		return array_keys($this->fields_for_model);
	}
	

	function is_a_valid_field_type($type){
		if(!$this->validator->is_a_nonempty_string_with_no_whitespace($type)) return false;
		return (bool) @$this->path_for_field_type($type);
	}
	
	function path_for_field_type($type){		
		if(!$this->validator->is_a_nonempty_string($type) || !$this->validator->is_a_string_with_no_whitespace($type)) return false;
		$valid_locations = array();
		if(!$this->property_is_empty('extension_directory'))
			$valid_locations[] = $this->extension_directory.'/field_markup_generators';
		$valid_locations[] = APPPATH.'libraries/form_markup_generator/field_markup_generators';	
		
		foreach($valid_locations as $directory){
			$path = $directory.'/'.$type.'.php';
			if(file_exists($path)) return $path;
		}

		return $this->error->should_be_a_valid_form_field_type($type);
	}
		
/////////
// OTHER FUNCTIONS
/////////
	function javascript_for_form(){
		if(in_array('rich_text_area', $this->field_types()) || in_array('tiny_rich_text_area', $this->field_types())){
			if(!file_exists('resources/js/ckeditor/ckeditor.js'))
				$this->error->warning("Ckeditor does not appear to be in the resources/js directory.  This means that rich text areas probably won't work.");
			
			return js_source_tag('resources/js/ckeditor/ckeditor.js');
		}
	}

	
}